const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const path = require('path');
const rateLimit = require('express-rate-limit');
const compression = require('compression');
require('dotenv').config();

const { initializeDatabase } = require('./config/db'); 
const authRoutes = require('./routes/auth');
const videoRoutes = require('./routes/videos');

const app = express();

// --- 1. Global Security ---
app.use(helmet({
    // Important: crossOriginResourcePolicy must be cross-origin for browsers to allow loading images/videos from a different domain (your frontend)
    crossOriginResourcePolicy: { policy: "cross-origin" },
    contentSecurityPolicy: {
        directives: {
            defaultSrc: ["'self'"],
            scriptSrc: ["'self'", "'unsafe-inline'"],
            styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
            // Allow images from your own API and data URIs
            imgSrc: ["'self'", "data:", "blob:", "*"], 
            // Allow videos from your own API
            videoSrc: ["'self'", "blob:", "*"], 
            connectSrc: ["'self'", "*"],
        },
    },
}));

// --- 2. Middlewares ---
app.use(compression());
// Allow CORS for your frontend URL
app.use(cors({ origin: [process.env.FRONTEND_URL, "http://localhost:3000", "http://localhost:5173","https://www.totorepublic.com","https://totorepublic.com", "https://api.totorepublic.com"], credentials: true }));
app.use(express.json({ limit: '10kb' }));
app.use(express.urlencoded({ extended: true, limit: '10kb' }));


app.use('/uploads', express.static(path.join(__dirname, 'uploads'), {
    maxAge: '1d',
    etag: true,
    setHeaders: (res) => {
        res.set('Access-Control-Allow-Origin', '*'); 
    }
}));

const limiter = rateLimit({
    windowMs: 15 * 60 * 1000,
    max: 100,
    message: "Too many requests, please try again later."
});
app.use('/api/', limiter);

app.use('/api/auth', authRoutes);
app.use('/api/videos', videoRoutes);

app.get('/', (req, res) => {
    res.send('Totorepublic API is Running...');
});

// Centralized Error Handler
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(err.status || 500).json({
        message: process.env.NODE_ENV === 'production' ? 'Internal server error' : err.message
    });
});

// --- 5. Boot Sequence ---
const PORT = process.env.PORT || 5000;

const startServer = async () => {
    try {
        await initializeDatabase(); 
        app.listen(PORT, () => {
            console.log(`🚀 Totorepublic Server running on port ${PORT}`);
        });
    } catch (error) {
        console.error('Failed to start server:', error.message);
        process.exit(1);
    }
};

startServer();