const mysql = require('mysql2');
require('dotenv').config();

const pool = mysql.createPool({
  host: 'localhost',
  user: process.env.DB_USER,
  database: process.env.DB_NAME,
  port: process.env.DB_PORT || 3306, 
  password: process.env.DB_PASSWORD,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0,
});

const db = pool.promise();

pool.getConnection((err, connection) => {
  if (err) {
    console.error('❌ Database Connection Failed:', err.message);
    return;
  }
  console.log('✅ Connected to MySQL Database.');
  connection.release();
});

const generateId = async (tableName, prefix) => {
  try {
    await db.query('START TRANSACTION');
    const [sequence] = await db.query('SELECT next_id FROM id_sequence WHERE table_name = ? FOR UPDATE', [tableName]);
    let nextId = sequence.length > 0 ? sequence[0].next_id : 1;
    const formattedId = `${prefix}${String(nextId).padStart(7, '0')}`;
    
    if (sequence.length > 0) {
      await db.query('UPDATE id_sequence SET next_id = ? WHERE table_name = ?', [nextId + 1, tableName]);
    } else {
      await db.query('INSERT INTO id_sequence (table_name, next_id) VALUES (?, ?)', [tableName, nextId + 1]);
    }
    
    await db.query('COMMIT');
    return formattedId;
  } catch (error) {
    await db.query('ROLLBACK');
    console.error(`ID Generation Error (${tableName}):`, error.message);
    throw error;
  }
};

const initializeDatabase = async () => {
  try {
    await db.query(`CREATE DATABASE IF NOT EXISTS ${process.env.DB_NAME}`);
    await db.query(`USE ${process.env.DB_NAME}`);

    await db.query(`
      CREATE TABLE IF NOT EXISTS id_sequence (
        table_name VARCHAR(50) PRIMARY KEY,
        next_id INT NOT NULL DEFAULT 1
      )
    `);

    const tables = [{ name: 'videos', prefix: 'VID' }, { name: 'admins', prefix: 'ADM' }];
    for (const table of tables) {
      await db.query('INSERT IGNORE INTO id_sequence (table_name, next_id) VALUES (?, ?)', [table.name, 1]);
    }

    await db.query(`
      CREATE TABLE IF NOT EXISTS videos (
        id VARCHAR(10) PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        category VARCHAR(50) DEFAULT 'General',
        creator VARCHAR(100),
        views VARCHAR(50),
        duration VARCHAR(20),
        thumbnail_path VARCHAR(255),
        video_path VARCHAR(255),
        is_premium BOOLEAN DEFAULT FALSE,
        is_trending BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);


    console.log('✅ Database Tables Initialized.');
  } catch (error) {
    console.error('❌ Init Error:', error.message);
    throw error;
  }
};

module.exports = { db, generateId, initializeDatabase };