const fs = require('fs');
const path = require('path');
const { generateId, db } = require('../config/db'); 

const rootDir = path.join(__dirname, '..');

const deleteFiles = (files) => {
    if (!files) return;
    const paths = [files.video?.[0]?.path, files.thumbnail?.[0]?.path];
    paths.forEach(p => { if (p && fs.existsSync(p)) fs.unlinkSync(p); });
};

const getRandomCreator = () => {
    const creators = ['Elite_Baddie', 'LagosSecret', 'Toto_Queen', 'Vibe_Master', 'X_Babe', 'Naija_Hottie', 'Private_VIP'];
    return creators[Math.floor(Math.random() * creators.length)];
};

const generateFakeViews = () => `${(Math.random() * (1.9 - 0.1) + 0.1).toFixed(1)}M`;

exports.getAllVideos = async (req, res) => {
    try {
        const { category, search } = req.query;
        let query = 'SELECT * FROM videos';
        let params = [];

        if (search) {
            query += ' WHERE MATCH(title, category) AGAINST(? IN NATURAL LANGUAGE MODE)';
            params.push(search);
            if (category) { query += ' AND category = ?'; params.push(category); }
        } else if (category) {
            query += ' WHERE category = ?';
            params.push(category);
        }

        query += ' ORDER BY created_at DESC';
        const [rows] = await db.execute(query, params);
        res.status(200).json({ success: true, data: rows });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
};

exports.getVideoById = async (req, res) => {
    try {
        const [rows] = await db.execute('SELECT * FROM videos WHERE id = ?', [req.params.id]);
        if (rows.length === 0) return res.status(404).json({ message: "Video not found" });
        res.status(200).json({ success: true, data: rows[0] });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
};

exports.uploadVideo = async (req, res) => {
    try {
        const { title, isPremium, isTrending, category, duration } = req.body;
        if (!req.files || !req.files['video'] || !req.files['thumbnail']) {
            return res.status(400).json({ message: "Both video and thumbnail are required" });
        }

        const videoId = await generateId('videos', 'VID');
        const videoDir = path.join(rootDir, 'uploads', 'videos');
        const thumbDir = path.join(rootDir, 'uploads', 'thumbnails');

        [videoDir, thumbDir].forEach(dir => { if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true }); });

        const videoExt = path.extname(req.files['video'][0].originalname);
        const thumbExt = path.extname(req.files['thumbnail'][0].originalname);
        const vName = `${videoId}${videoExt}`;
        const tName = `${videoId}${thumbExt}`;

        fs.renameSync(req.files['video'][0].path, path.join(videoDir, vName));
        fs.renameSync(req.files['thumbnail'][0].path, path.join(thumbDir, tName));

        await db.execute(
            `INSERT INTO videos (id, title, category, creator, views, duration, thumbnail_path, video_path, is_premium, is_trending) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [videoId, title, category || 'Hotties', getRandomCreator(), generateFakeViews(), duration || '0:00', `uploads/thumbnails/${tName}`, `uploads/videos/${vName}`, isPremium === 'true', isTrending === 'true']
        );

        res.status(201).json({ success: true, videoId, message: "Upload successful" });
    } catch (error) {
        deleteFiles(req.files);
        res.status(500).json({ error: "Upload failed", details: error.message });
    }
};

exports.deleteVideo = async (req, res) => {
    try {
        const [rows] = await db.execute('SELECT video_path, thumbnail_path FROM videos WHERE id = ?', [req.params.id]);
        if (rows.length === 0) return res.status(404).json({ message: "Video not found" });

        [rows[0].video_path, rows[0].thumbnail_path].forEach(p => {
            const fullPath = path.join(rootDir, p);
            if (fs.existsSync(fullPath)) fs.unlinkSync(fullPath);
        });

        await db.execute('DELETE FROM videos WHERE id = ?', [req.params.id]);
        res.json({ success: true, message: "Deleted successfully" });
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
};